# Browsershot PDF Generation Setup

## Overview

This project now uses **Spatie Browsershot** for PDF generation instead of DomPDF. Browsershot uses Chrome/Chromium to render PDFs, providing better CSS support and more accurate rendering.

## Architecture

### Class Structure

```
QuoteGenerator (Main Service)
    ├── QuoteDataProvider (Data Preparation)
    └── BrowsershotPdfGenerator (PDF Generation)
```

### Files Created

1. **QuoteDataProvider.php** - `/plugins/majormedia/project/classes/QuoteDataProvider.php`
   - Prepares all data needed for quote views
   - Library-agnostic (can be used with any PDF generator)
   - Handles logo paths, pricing, project data, etc.

2. **BrowsershotPdfGenerator.php** - `/plugins/majormedia/project/classes/BrowsershotPdfGenerator.php`
   - Wrapper for Spatie Browsershot
   - Provides consistent PDF configuration
   - Methods: `generateFromView()`, `generateFromHtml()`, `saveFromView()`, `saveFromHtml()`

3. **QuoteGenerator.php** - `/plugins/majormedia/project/classes/QuoteGenerator.php`
   - Main orchestration service
   - Handles quote record creation
   - Coordinates data preparation and PDF generation
   - Attaches PDFs to Quote model

### Trigger Flow

```
Project Model: is_content_required changed to true
    ↓
Project::afterUpdate() hook
    ↓
QuoteGenerator::generateQuotesForProject()
    ↓
For each FrontDesign (Template/Custom):
    ├── QuoteDataProvider::getData() → Prepare view data
    ├── BrowsershotPdfGenerator::generateFromView() → Render HTML & Generate PDF
    ├── Save PDF to temp file
    ├── Create File model instance
    └── Attach to Quote record
```

## Installation

### 1. Install Browsershot Package

```bash
composer require spatie/browsershot
```

### 2. Install Node.js and Puppeteer

Browsershot requires Node.js and Puppeteer (headless Chrome).

#### Option A: Install Puppeteer locally (Recommended for development)

```bash
npm install puppeteer
```

#### Option B: Use system Chrome/Chromium

```bash
# macOS
brew install chromium

# Ubuntu/Debian
sudo apt-get install chromium-browser

# CentOS/RHEL
sudo yum install chromium
```

### 3. Verify Installation

Test if Browsershot can find Chrome:

```bash
php artisan tinker
```

```php
// In tinker
use Spatie\Browsershot\Browsershot;
Browsershot::html('<h1>Test</h1>')->pdf();
// Should return binary PDF data without errors
```

## Configuration

### Browsershot Options (Already Configured)

The `BrowsershotPdfGenerator` class is pre-configured with:

- **Format**: A4
- **Orientation**: Portrait
- **Margins**: 0mm (all sides)
- **Print Background**: Enabled (for colors and images)
- **Wait Until Network Idle**: Enabled (loads external resources)
- **Timeout**: 60 seconds

### Custom Configuration

To modify PDF options, edit the options array in `QuoteGenerator::generatePdf()`:

```php
$pdfContent = $this->pdfGenerator->generateFromView(
    'majormedia.project::quote.quote',
    $data,
    [
        'format' => 'A4',           // A4, Letter, Legal, etc.
        'orientation' => 'portrait', // portrait or landscape
        'margin_top' => 10,          // in mm
        'margin_right' => 10,
        'margin_bottom' => 10,
        'margin_left' => 10,
        'print_background' => true,  // Print colors and images
        'scale' => 1.0,              // Page scale (0.1 to 2)
        'timeout' => 60              // Timeout in seconds
    ]
);
```

## Usage

### Automatic Generation

Quotes are automatically generated when:
- A project's `is_content_required` field is updated to `true`
- The `Project::afterUpdate()` hook triggers `QuoteGenerator`

### Manual Generation

```php
use MajorMedia\Project\Classes\QuoteGenerator;
use MajorMedia\Project\Models\Project;

$project = Project::find($projectId);
$generator = new QuoteGenerator();
$quotes = $generator->generateQuotesForProject($project);

// Or generate for specific design
$quote = $generator->generateQuote($project, $frontDesignId);
```

### Direct PDF Generation

```php
use MajorMedia\Project\Classes\BrowsershotPdfGenerator;

$generator = new BrowsershotPdfGenerator();

// From view
$pdfContent = $generator->generateFromView(
    'majormedia.project::quote.quote',
    ['data' => 'value']
);

// From HTML
$html = '<html><body><h1>Test</h1></body></html>';
$pdfContent = $generator->generateFromHtml($html);

// Save to file
$generator->saveFromView(
    'majormedia.project::quote.quote',
    ['data' => 'value'],
    '/path/to/output.pdf'
);
```

## Logging

The system logs detailed information at each step:

- 🚀 Starting quote generation
- 💾 Quote record saved
- 📄 Preparing data
- 🎨 Rendering Blade view
- 📦 PDF generated with size
- ✅ Success messages
- ❌ Error messages with stack traces

Check logs at: `storage/logs/system.log`

## Advantages Over DomPDF

### 1. **Better CSS Support**
   - Full CSS3 support (flexbox, grid, transforms, etc.)
   - Modern CSS properties work out of the box
   - No need to avoid specific CSS properties

### 2. **Accurate Rendering**
   - Uses real Chrome engine
   - What you see in browser = what you get in PDF
   - Better image and font handling

### 3. **External Resources**
   - Can load external stylesheets
   - Can load web fonts (Google Fonts, etc.)
   - Can load images from URLs

### 4. **JavaScript Support**
   - Can execute JavaScript before rendering
   - Useful for dynamic content generation

## Troubleshooting

### Issue: "Could not find Chrome"

**Solution**: Install Puppeteer or set Chrome path manually:

```php
Browsershot::html($html)
    ->setChromePath('/path/to/chrome')
    ->pdf();
```

### Issue: Timeout errors

**Solution**: Increase timeout in options:

```php
['timeout' => 120] // 2 minutes
```

### Issue: Images not loading

**Solution**: Ensure `wait_until_network_idle` is enabled:

```php
['wait_until_network_idle' => true]
```

Or use base64 encoded images in your views.

### Issue: Fonts not rendering

**Solution**: Include web fonts in your Blade template:

```html
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@400;700&display=swap" rel="stylesheet">
```

## File Locations

```
plugins/majormedia/project/
├── classes/
│   ├── QuoteGenerator.php          (Main service)
│   ├── BrowsershotPdfGenerator.php (PDF wrapper)
│   └── QuoteDataProvider.php       (Data provider)
├── models/
│   └── Project.php                  (afterUpdate hook)
└── views/
    └── quote/
        ├── app.blade.php            (Layout template)
        └── quote.blade.php          (Content template)
```

## Testing

Test the generation manually:

```bash
php artisan tinker
```

```php
$project = \MajorMedia\Project\Models\Project::find(1);
$generator = new \MajorMedia\Project\Classes\QuoteGenerator();
$quotes = $generator->generateQuotesForProject($project);
print_r($quotes);
```

## Next Steps

1. Install Browsershot: `composer require spatie/browsershot`
2. Install Puppeteer: `npm install puppeteer`
3. Test generation with a sample project
4. Check logs for any errors
5. Adjust PDF options if needed

## Support

For issues with Browsershot, check:
- [Browsershot Documentation](https://github.com/spatie/browsershot)
- [Puppeteer Documentation](https://pptr.dev/)
