# Quote Generation Button Implementation

## Overview

Added a "Générer devis" (Generate Quote) button to the Project list columns that opens a popup form to collect quote information before generating the PDF, eliminating the dependency on the `is_content_required` field.

## Implementation Summary

### 1. Files Created

#### `/plugins/majormedia/project/models/project/_generate_quote_button.htm`
Button partial displayed in the Actions column of the Project list.
- Triggers a popup when clicked
- Passes project_id to the handler

#### `/plugins/majormedia/project/controllers/projects/_generate_quote_popup.htm`
Modal popup form with the following fields:
- **Client Name** (required) - Pre-filled with project user's name
- **Quote Date** (required) - Defaults to current date
- **Front Design Type** (required) - Dropdown with active front designs
- **Notes/Comments** (optional) - Text area for additional information

#### `/plugins/majormedia/project/updates/add_notes_to_quotes.php`
Database migration to add the `notes` field to the `majormedia_project_quotes` table.

### 2. Files Modified

#### `/plugins/majormedia/project/models/project/columns.yaml`
Added new column definition:
```yaml
generate_quote:
    label: Actions
    type: partial
    path: $/majormedia/project/models/project/_generate_quote_button.htm
    sortable: false
    width: 120
```

#### `/plugins/majormedia/project/controllers/Projects.php`
Added two AJAX handler methods:

**`onLoadGenerateQuoteForm()`**
- Loads the popup form
- Fetches project and active front designs
- Pre-fills form with project data

**`onGenerateQuote()`**
- Validates form inputs
- Generates quote using `QuoteGenerator` class
- Updates quote with form data (client_name, quote_date, notes)
- Returns success/error response
- Refreshes the list after generation

#### `/plugins/majormedia/project/models/Quote.php`
Added `notes` to the `$fillable` array:
```php
protected $fillable = [
    'id',
    'project_id',
    'quote_date',
    'client_name',
    'reference',
    'recomanded_amount',
    'final_amount',
    'notes',  // NEW
];
```

#### `/plugins/majormedia/project/updates/version.yaml`
Added new version entry:
```yaml
v1.0.21:
    - "Add notes field to quotes table"
    - add_notes_to_quotes.php
```

## How It Works

### User Flow

1. **Navigate to Projects List**: User sees "Actions" column with "Générer devis" button
2. **Click Button**: Popup form opens with pre-filled data
3. **Fill Form**: User reviews/edits:
   - Client name (pre-filled from project user)
   - Quote date (defaults to today)
   - Front design type (selects from dropdown)
   - Optional notes
4. **Submit**: Click "Générer le devis" button
5. **Generation**: System generates quote PDF using Spatie Browsershot
6. **Confirmation**: Success message displays and list refreshes

### Technical Flow

```
User clicks "Générer devis"
        ↓
onLoadGenerateQuoteForm() handler
        ↓
Loads popup with form
        ↓
User fills form and submits
        ↓
onGenerateQuote() handler
        ↓
Validates inputs
        ↓
Calls QuoteGenerator::generateQuote()
        ↓
Creates/updates Quote record
        ↓
Generates PDF using Browsershot
        ↓
Attaches PDF to Quote
        ↓
Updates quote with form data
        ↓
Returns success response
        ↓
Refreshes list and shows flash message
```

## Database Schema Changes

### New Field: `notes`
- **Table**: `majormedia_project_quotes`
- **Type**: `text`
- **Nullable**: `yes`
- **Position**: After `quote_date`
- **Purpose**: Store additional notes/comments from the quote generation form

## Installation

### 1. Run Migration

When the database server is running, execute:

```bash
php artisan october:up
```

OR

```bash
php artisan plugin:refresh MajorMedia.Project
```

### 2. Clear Cache

```bash
php artisan cache:clear
```

## Features

✅ **On-Demand Quote Generation**: Generate quotes anytime without relying on field updates
✅ **Custom Client Name**: Override the default project user name
✅ **Flexible Date Selection**: Choose any quote date
✅ **Design Type Selection**: Select specific front design (Template or Custom)
✅ **Additional Notes**: Add context or special instructions
✅ **Real-time Feedback**: Success/error messages with automatic list refresh
✅ **Validation**: All required fields validated before generation
✅ **Error Handling**: Comprehensive try-catch with logging

## Benefits Over Previous Method

### Before (is_content_required field)
- ❌ Required toggling a field in the project form
- ❌ Generated quotes for ALL active front designs automatically
- ❌ No control over client name or date
- ❌ No way to add notes
- ❌ Difficult to regenerate quotes

### After (Generate Button)
- ✅ Direct action from list view
- ✅ Generate quote for specific front design
- ✅ Full control over quote details
- ✅ Can add notes/comments
- ✅ Easy to regenerate quotes anytime
- ✅ Better user experience with modal form

## Usage Example

```php
// The button automatically appears in the Actions column
// Clicking it triggers the popup

// After form submission, the system:
1. Creates Quote record with reference number (DV-YYYYMM-XXXX)
2. Generates PDF using QuoteGenerator
3. Attaches PDF to Quote
4. Updates with form data
5. Shows success message
```

## Future Enhancements

Potential improvements:
- [ ] Add ability to select multiple front designs at once
- [ ] Preview quote before final generation
- [ ] Email quote directly to client
- [ ] Duplicate existing quote
- [ ] Version control for quote revisions
- [ ] Custom reference number format

## Troubleshooting

### Button doesn't appear
- Clear cache: `php artisan cache:clear`
- Check partial file exists: `_generate_quote_button.htm`
- Verify column definition in `columns.yaml`

### Popup doesn't open
- Check browser console for JavaScript errors
- Verify AJAX handler exists in controller
- Ensure October CMS AJAX framework is loaded

### Quote generation fails
- Check logs: `storage/logs/system.log`
- Verify Browsershot is installed
- Ensure project has required relationships (sitemap_pages, features, etc.)
- Check front design is active

### Migration fails
- Start database server (XAMPP)
- Run: `php artisan october:up`
- Check migration file syntax

## Related Files

- **QuoteGenerator**: `/plugins/majormedia/project/classes/QuoteGenerator.php`
- **BrowsershotPdfGenerator**: `/plugins/majormedia/project/classes/BrowsershotPdfGenerator.php`
- **QuoteDataProvider**: `/plugins/majormedia/project/classes/QuoteDataProvider.php`
- **Quote Template**: `/plugins/majormedia/project/views/quote/quote.blade.php`
- **Layout Template**: `/plugins/majormedia/project/views/quote/app.blade.php`

## Notes

- The old `is_content_required` field logic remains in the Project model's `afterUpdate()` method
- This implementation provides an alternative, more flexible approach
- Both methods can coexist if needed
- The `notes` field is optional and can be left blank
