<?php
namespace Majormedia\Project\Http;

ini_set("memory_limit", "-1");

use Backend\Classes\Controller;
use MajorMedia\Project\Models\SocialPlatform;
use MajorMedia\ToolBox\Traits\JsonAbort;
use MajorMedia\Project\Models\ProjectType;
use MajorMedia\ToolBox\Traits\RetrieveUser;
use Majormedia\ToolBox\Traits\GetFileFromBase64;
use MajorMedia\ToolBox\Traits\GetValidatedInput;
use MajorMedia\ToolBox\Utility\ErrorCodes;
/**
 * Projects Back-end Controller
 */
class Projects extends Controller
{
    use RetrieveUser, GetValidatedInput, JsonAbort, GetFileFromBase64;
    public $implement = [
        'MajorMedia.ToolBox.Behaviors.RestController'
    ];

    public $restConfig = 'config_rest.yaml';

    public function extendModel($model)
    {
        $this->retrieveUser();
        $user = $this->user;
        return $model->where('user_id', $user->id);
    }
    public function store()
    {
        try {
            // Retrieve the current user
            $this->retrieveUser();
            $user = $this->user;

            // Validate and retrieve input data
            $data = $this->getValidatedInput(
                [
                    'name',
                    'domain',
                    'logo',
                    'is_logo_required',
                    'is_multi_lang',
                    'is_content_required',
                    'country_name',
                    'languages',
                    'address',
                    'email',
                    'phone',
                    'working_hour',
                    'social_platforms',
                    'type_id',
                    'nbr_products',
                    'nbr_categories',
                    'nbr_brands',
                    'sales_method',
                    'carrier',
                    'delivery_method',
                    'shipping_parameters',
                    'payment_methods'
                ],
                [
                    'name' => 'nullable|string',
                    'domain' => 'nullable|string',
                    'logo' => 'nullable|string',
                    'is_logo_required' => 'nullable|boolean',
                    'is_multi_lang' => 'nullable|boolean',
                    'is_content_required' => 'nullable|boolean',
                    'country_name' => 'nullable|string',
                    'languages' => 'nullable|array',
                    'address' => 'nullable|string',
                    'email' => 'nullable|string',
                    'phone' => 'nullable|string',
                    'working_hour' => 'nullable|string',
                    'social_platforms' => 'nullable|array',
                    'social_platforms.*.id' => 'nullable|numeric',
                    'social_platforms.*.url' => 'nullable|string',
                    'type_id' => 'nullable|numeric',
                    'nbr_products' => 'nullable|string',
                    'nbr_categories' => 'nullable|integer',
                    'nbr_brands' => 'nullable|integer',
                    'sales_method' => 'nullable|in:1,2',
                    'carrier' => 'nullable|string|max:255',
                    'delivery_method' => 'nullable|in:1,2',
                    'shipping_parameters' => 'nullable|array',
                    'payment_methods' => 'nullable|array'
                ]
            );

            if (!ProjectType::find($data['type_id'])) {
                return $this->JsonAbort([
                    'status' => 'error',
                    'error' => ErrorCodes::PROJECT_TYPE_NOT_VALID,
                ], 400);
            }

            if (!empty($data['social_platforms'])) {
                foreach ($data['social_platforms'] as $platform) {
                    if (!SocialPlatform::find($platform['id'])) {
                        return $this->JsonAbort([
                            'status' => 'error',
                            'error' => ErrorCodes::SOCIAL_PLATFORM_NOT_VALID,
                        ], 400);
                    }
                }
            }

            $logo = $data['logo'];
            $socialPlatforms = $data['social_platforms'];
            $paymentMethods = $data['payment_methods'];
            $shippingParameters = $data['shipping_parameters'];
            unset($data['logo']);
            unset($data['social_platforms']);
            unset($data['payment_methods']);
            unset($data['shipping_parameters']);


            if ($data['is_content_required'] == null) {
                unset($data['is_content_required']);
            }

            $project = $user->project()->create($data);

            if ($logo) {
                $logoAsFile = $this->getFileFromBase64($logo);
                $project->logo = $logoAsFile;
            }
            $project->save();

            if (isset($paymentMethods)) {
                foreach ($paymentMethods as $payment)
                    $project->payment_methods()->attach($payment);
            }
            if (isset($shippingParameters)) {
                foreach ($shippingParameters as $shippingParameter)
                    $project->shipping_parameters()->attach($shippingParameter);
            }

            if (!empty($socialPlatforms)) {
                foreach ($socialPlatforms as $platform) {
                    $project->socialPlatforms()->attach($platform['id'], ['url' => $platform['url']]);
                }
            }

            return $this->JsonAbort([
                'status' => 'success',
            ], 201);
        } catch (\Exception $e) {
            // Handle exceptions
            return $this->JsonAbort([
                'status' => 'error',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function update($id)
    {
        $this->retrieveUser();
        $this->retrieveUser();
        $user = $this->user;
        // List of all possible fields
        $fields = [
            'name',
            'domain',
            'logo',
            'is_logo_required',
            'is_multi_lang',
            'is_content_required',
            'country_name',
            'languages',
            'address',
            'email',
            'phone',
            'working_hour',
            'social_platforms',
            'type_id',
            'nbr_products',
            'nbr_categories',
            'nbr_brands',
            'sales_method',
            'carrier',
            'delivery_method',
            'shipping_parameters',
            'payment_methods'
        ];

        // Validation rules
        $rules = [
            'name' => 'nullable|string',
            'domain' => 'nullable|string',
            'logo' => 'nullable|string',
            'is_logo_required' => 'nullable|boolean',
            'is_multi_lang' => 'nullable|boolean',
            'is_content_required' => 'nullable|boolean',
            'country_name' => 'nullable|string',
            'languages' => 'nullable|array',
            'address' => 'nullable|string',
            'email' => 'nullable|string',
            'phone' => 'nullable|string',
            'working_hour' => 'nullable|string',
            'social_platforms' => 'nullable|array',
            'social_platforms.*.id' => 'nullable|numeric',
            'social_platforms.*.url' => 'nullable|string',
            'type_id' => 'nullable|numeric',
            'nbr_products' => 'nullable|string',
            'nbr_categories' => 'nullable|integer',
            'nbr_brands' => 'nullable|integer',
            'sales_method' => 'nullable|in:1,2',
            'carrier' => 'nullable|string|max:255',
            'delivery_method' => 'nullable|in:1,2',
            'shipping_parameters' => 'nullable|array',
            'payment_methods' => 'nullable|array'
        ];

        $filteredFields = array_intersect_key($rules, request()->all());

        $data = $this->getValidatedInput(array_keys($filteredFields), $filteredFields);


        $logo = $data['logo'] ?? null;
        $socialPlatforms = $data['social_platforms'] ?? null;
        $paymentMethods = $data['payment_methods'] ?? null;
        $shippingParameters = $data['shipping_parameters'] ?? null;

        unset($data['logo'], $data['social_platforms'], $data['payment_methods'], $data['shipping_parameters']);

        if (isset($data['is_content_required']) && $data['is_content_required'] == null) {
            unset($data['is_content_required']);
        }

        $project = $user->project;
        if ($project) {
            $filteredData = array_filter($data, function ($value) {
                return !is_null($value);
            });
            $project->update($filteredData);
        }
        if ($logo) {
            $logoAsFile = $this->getFileFromBase64($logo);
            $project->logo = $logoAsFile;
        }
        if (isset($paymentMethods)) {
            $project->payment_methods()->sync($paymentMethods);
        }

        if (isset($shippingParameters)) {

            $project->shipping_parameters()->sync($shippingParameters);
        }

        if (!empty($socialPlatforms)) {
            $syncData = [];

            foreach ($socialPlatforms as $platform) {
                $syncData[$platform['id']] = ['url' => $platform['url']];
            }
            $project->socialPlatforms()->sync($syncData);
        }

        return $this->JsonAbort([
            'status' => 'success',
        ], 200);
    }

    public function destroy($id)
    {
        try {
            $this->retrieveUser();
            $user = $this->user;
            $project = $user->projects()->findOrFail($id);
            $project->delete();
            return $this->JsonAbort([
                'status' => 'success',
            ], 200);
        } catch (\Exception $e) {
            return $this->JsonAbort([
                'status' => 'error',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    /**
     * This is a Method will update all the files of the project
     */
    public function updateFilesProject()
    {
        $this->retrieveUser();
        $user = $this->user;
        $project = $user->project;

        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND
            ], 200);
        }

        $data = $this->getValidatedInput([
            'products_content',
            'products_images',
            'categories_content',
            'categories_images',
            'brands_content',
            'brands_images',
            'legal_document',
            'privacy_policy',
            'terms_and_conditions',
        ], [
            'products_content' => 'nullable|string',
            'products_images' => 'nullable|string',
            'categories_content' => 'nullable|string',
            'categories_images' => 'nullable|string',
            'brands_content' => 'nullable|string',
            'brands_images' => 'nullable|string',
            'legal_document' => 'nullable|string',
            'privacy_policy' => 'nullable|string',
            'terms_and_conditions' => 'nullable|string',
        ]);

        $fileFields = [
            'products_content',
            'products_images',
            'categories_content',
            'categories_images',
            'brands_content',
            'brands_images',
            'legal_document',
            'privacy_policy',
            'terms_and_conditions',
        ];

        foreach ($fileFields as $field) {
            if (!empty($data[$field])) {
                $file = $this->getFileFromBase64($data[$field]);
                $project->{$field} = $file;
            }
        }

        $project->save();

        return $this->JsonAbort([
            'status' => 'success'
        ], 200);
    }

    public function legalFiles()
    {
        $this->retrieveUser();
        $user = $this->user;
        $project = $user->project;

        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND
            ], 200);
        }
        // Get only the legal files
        $legalFiles = [
            'legal_document' => $project->legal_document->path,
            'privacy_policy' => $project->privacy_policy->path,
            'terms_and_conditions' => $project->terms_and_conditions->path
        ];

        return $this->JsonAbort([
            'status' => 'success',
            'data' => $legalFiles,
        ], 200);
    }
    public function ecommerceFiles()
    {
        $this->retrieveUser();
        $user = $this->user;

        $project = $user->project;

        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND
            ], 200);
        }
        $ecommerceFiles = [
            'products_content' => $project->products_content->path,
            'products_images' => $project->products_images->path,
            'categories_content' => $project->categories_content->path,
            'categories_images' => $project->categories_images->path,
            'brands_content' => $project->brands_content->path,
            'brands_images' => $project->brands_images->path
        ];

        return $this->JsonAbort([
            'status' => 'success',
            'data' => $ecommerceFiles
        ], 200);
    }

}
