<?php
namespace Majormedia\Structure\Http;

use Backend\Classes\Controller;
use MajorMedia\Structure\Models\Feature;
use MajorMedia\ToolBox\Traits\JsonAbort;
use MajorMedia\ToolBox\Traits\GetValidatedInput;
use MajorMedia\ToolBox\Traits\RetrieveUser;
use MajorMedia\ToolBox\Utility\ErrorCodes;
use Illuminate\Support\Facades\Session;
/**
 * Features Back-end Controller
 */
class Features extends Controller
{
    use RetrieveUser, JsonAbort, GetValidatedInput;
    public $implement = [
        'MajorMedia.ToolBox.Behaviors.RestController'
    ];

    public $restConfig = 'config_rest.yaml';

    // public function extendModel($model)
    // {
    //     Session::put('required_token', true);
    //     $this->retrieveUser();
    //     $user = $this->user;

    //     if (!$user->project || !$user->project->project_type) {
    //         throw new \Exception('User project or project type not found.');
    //     }
    //     $projectId = $user->project->id;
    //     $projectTypeId = $user->project->project_type->id;
    //     $features = $model->whereHas('project_types', function ($query) use ($projectTypeId) {
    //         $query->where('project_type_id', $projectTypeId);
    //     })->with(['project_types'])->get();

    //     $customFeatures = $model->whereHas('projects', function ($query) use ($projectId) {
    //         $query->where('project_id', $projectId);
    //     })->with(['projects'])->get();

    //     return $this->JsonAbort([
    //         'status' => 'success',
    //         'data' => $features,
    //         'customFeatures' => $customFeatures,
    //     ], 200);
    // }

    public function extendModel($model)
    {
        Session::put('required_token', true);
        $this->retrieveUser();
        $user = $this->user;

        if (!$user->project || !$user->project->project_type) {
            throw new \Exception('User project or project type not found.');
        }

        $projectId = $user->project->id;
        $projectTypeId = $user->project->project_type->id;

        //this query get the features linked with project type
        $features = $model->whereHas('project_types', function ($query) use ($projectTypeId) {
            $query->where('project_type_id', $projectTypeId);
        })->with([
                    'project_types' => function ($query) {
                        $query->withPivot(['is_required', 'is_primary']);
                    }
                ])->get();

        //format the feature to containe the pivot data
        $formattedFeatures = $features->map(function ($feature) {
            $projectType = $feature->project_types->first();

            return [
                'id' => $feature->id,
                'title' => $feature->title,
                'description' => $feature->description,
                'price' => $feature->price,
                'isEnabled' => $feature->isEnabled,
                'project_type' => $projectType ? [
                    'id' => $projectType->id,
                    'name' => $projectType->name,
                    'pivot' => [
                        'is_required' => $projectType->pivot->is_required,
                        'is_primary' => $projectType->pivot->is_primary
                    ]
                ] : null,
            ];
        });

        //This query get the features that was created and linked direclty with project
        $customFeatures = $model->whereHas('project', function ($query) use ($projectId) {
            $query->where('id', $projectId);
        })->get();

        return $this->JsonAbort([
            'status' => 'success',
            'data' => $formattedFeatures,
            'customFeatures' => $customFeatures,
        ], 200);
    }

    public function store()
    {
        $this->retrieveUser();

        $user = $this->user;

        $project = $user->project;

        $projectId = $project->id;

        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND,
            ], 404);
        }

        $data = $this->getValidatedInput([
            'title',
            'description',
        ], [
            'title' => 'nullable|string',
            'description' => 'nullable|string'
        ]);

        $data['project_id'] = $projectId;

        $newFeature = Feature::create($data);

        return $this->JsonAbort([
            'status' => 'success',
        ], 200);
    }

    /**
     * this function will sync the features with the project
     */
    public function syncFeaturesWithProject()
    {
        $this->retrieveUser();
        $user = $this->user;

        $project = $user->project;
        if (!$project || !$project->project_type) {
            throw new \Exception('User project or project type not found.');
        }

        $data = $this->getValidatedInput([
            'features'
        ], [
            'features' => 'nullable|array',
            'features.*' => 'integer'
        ]);

        $project->features()->sync($data['features'] ?? []);

        return $this->JsonAbort([
            'status' => 'success'
        ], 200);
    }


}
