<?php
/**
 * Debug script for testing quote generation
 *
 * Usage on server:
 * php test_quote_generation.php <project_id>
 *
 * Example:
 * php test_quote_generation.php 1
 */

// Bootstrap October CMS
require __DIR__ . '/bootstrap/autoload.php';
$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use MajorMedia\Project\Models\Project;
use MajorMedia\Project\Models\FrontDesign;
use MajorMedia\Project\Models\Quote;
use MajorMedia\Project\Classes\QuoteGenerator;

// Get project ID from command line
$projectId = $argv[1] ?? null;

if (!$projectId) {
    echo "❌ Error: Please provide a project ID\n";
    echo "Usage: php test_quote_generation.php <project_id>\n";
    exit(1);
}

echo "🔍 Testing Quote Generation System\n";
echo "==================================\n\n";

// Test 1: Check if project exists
echo "1️⃣  Checking if project exists...\n";
$project = Project::find($projectId);
if (!$project) {
    echo "   ❌ Project not found with ID: {$projectId}\n";
    exit(1);
}
echo "   ✅ Project found: {$project->name} (ID: {$project->id})\n";
echo "   - Type: " . ($project->project_type->name ?? 'N/A') . "\n";
echo "   - is_content_required: " . ($project->is_content_required ? 'TRUE' : 'FALSE') . "\n\n";

// Test 2: Check FrontDesigns
echo "2️⃣  Checking FrontDesigns...\n";
$frontDesigns = FrontDesign::where('is_active', 1)->get();
echo "   Found " . $frontDesigns->count() . " active front designs\n";
if ($frontDesigns->isEmpty()) {
    echo "   ❌ No active front designs found!\n";
    echo "   Please activate front designs in the database:\n";
    echo "   UPDATE majormedia_project_front_designs SET is_active = 1;\n";
    exit(1);
}
foreach ($frontDesigns as $fd) {
    echo "   - {$fd->name} (ID: {$fd->id})\n";
}
echo "\n";

// Test 3: Check pivot table prices
echo "3️⃣  Checking pricing in pivot table...\n";
$prices = DB::table('majormedia_project_type_designs')
    ->where('project_type_id', $project->type_id)
    ->get();

if ($prices->isEmpty()) {
    echo "   ❌ No prices found for project type {$project->type_id}!\n";
    echo "   Please add prices to majormedia_project_type_designs table\n";
    exit(1);
}

foreach ($prices as $price) {
    $fdName = FrontDesign::find($price->front_design_id)->name ?? 'Unknown';
    echo "   - {$fdName}: {$price->initial_price} MAD\n";
}
echo "\n";

// Test 4: Check pages
echo "4️⃣  Checking sitemap pages...\n";
$pageCount = $project->sitemap_pages()->count();
echo "   Found {$pageCount} pages\n\n";

// Test 5: Check features
echo "5️⃣  Checking features...\n";
$features = $project->features()->wherePivot('project_id', $project->id)->get();
$customFeatures = $project->custom_features()->get();
$allFeatures = $features->merge($customFeatures);
echo "   Found " . $allFeatures->count() . " features\n";
$totalFeaturesCost = 0;
foreach ($allFeatures as $feature) {
    $price = $feature->price ?? 0;
    $totalFeaturesCost += $price;
    echo "   - {$feature->title}: {$price} MAD\n";
}
echo "   Total features cost: {$totalFeaturesCost} MAD\n\n";

// Test 6: Test quote generation
echo "6️⃣  Testing quote generation...\n";
try {
    $generator = new QuoteGenerator();

    echo "   Generating quotes for project {$project->id}...\n";
    $quotes = $generator->generateQuotesForProject($project);

    if (empty($quotes)) {
        echo "   ❌ No quotes were generated!\n";
        echo "   Check the logs: tail -f storage/logs/system.log\n";
        exit(1);
    }

    echo "   ✅ Generated " . count($quotes) . " quotes successfully!\n\n";

    // Display quote details
    foreach ($quotes as $quote) {
        echo "   Quote ID: {$quote->id}\n";
        echo "   - Reference: {$quote->reference}\n";
        echo "   - Front Design: " . ($quote->front_design->name ?? 'N/A') . "\n";
        echo "   - Total HT: {$quote->recomanded_amount} MAD\n";
        echo "   - Total TTC: {$quote->final_amount} MAD\n";
        echo "   - PDF attached: " . ($quote->quote ? 'YES ✅' : 'NO ❌') . "\n";
        if ($quote->quote) {
            echo "   - PDF path: {$quote->quote->path}\n";
        }
        echo "\n";
    }

} catch (\Exception $e) {
    echo "   ❌ Error: " . $e->getMessage() . "\n";
    echo "   Stack trace:\n";
    echo $e->getTraceAsString() . "\n";
    exit(1);
}

// Test 7: Check database for created quotes
echo "7️⃣  Verifying quotes in database...\n";
$dbQuotes = Quote::where('project_id', $project->id)->get();
echo "   Found " . $dbQuotes->count() . " quotes in database for this project\n";
foreach ($dbQuotes as $q) {
    echo "   - Quote #{$q->id}: {$q->reference} ({$q->front_design->name ?? 'N/A'})\n";
}
echo "\n";

echo "✅ All tests completed successfully!\n";
echo "\nYou can now view the quotes in the admin panel:\n";
echo "Navigate to: Projects > Quotes\n";
